// ===== BONIFLIX APP JS =====
(function () {
  'use strict';

  // ===== Navbar scroll effect =====
  const navbar = document.querySelector('.navbar');
  if (navbar) {
    window.addEventListener('scroll', () => {
      navbar.classList.toggle('scrolled', window.scrollY > 10);
    });
  }

  // ===== Mobile menu =====
  const menuToggle = document.getElementById('menuToggle');
  const mobileMenu = document.getElementById('mobileMenu');
  if (menuToggle && mobileMenu) {
    menuToggle.addEventListener('click', () => {
      mobileMenu.classList.toggle('open');
      const spans = menuToggle.querySelectorAll('span');
      if (mobileMenu.classList.contains('open')) {
        spans[0].style.transform = 'rotate(45deg) translate(5px, 5px)';
        spans[1].style.opacity = '0';
        spans[2].style.transform = 'rotate(-45deg) translate(5px, -5px)';
      } else {
        spans[0].style.transform = '';
        spans[1].style.opacity = '';
        spans[2].style.transform = '';
      }
    });
    document.addEventListener('click', (e) => {
      if (!menuToggle.contains(e.target) && !mobileMenu.contains(e.target)) {
        mobileMenu.classList.remove('open');
        const spans = menuToggle.querySelectorAll('span');
        spans.forEach(s => { s.style.transform = ''; s.style.opacity = ''; });
      }
    });
  }

  // ===== Page loader =====
  window.addEventListener('load', () => {
    const loader = document.getElementById('pageLoader');
    if (loader) {
      // Show loader for at least 800ms so logo animation is visible
      setTimeout(() => {
        loader.classList.add('hidden');
        setTimeout(() => loader.remove(), 520);
      }, 800);
    }
  });

  // ===== Row arrow scrolling =====
  document.querySelectorAll('.drama-row-wrap').forEach(wrap => {
    const row = wrap.querySelector('.drama-row');
    const leftBtn = wrap.querySelector('.row-arrow.left');
    const rightBtn = wrap.querySelector('.row-arrow.right');
    if (!row) return;
    const scrollAmount = () => row.offsetWidth * 0.75;
    if (leftBtn) leftBtn.addEventListener('click', () => row.scrollBy({ left: -scrollAmount(), behavior: 'smooth' }));
    if (rightBtn) rightBtn.addEventListener('click', () => row.scrollBy({ left: scrollAmount(), behavior: 'smooth' }));
  });

  // ===== Lazy image loading =====
  function lazyLoadImages() {
    const imgs = document.querySelectorAll('img[data-src]');
    if (!imgs.length) return;
    const obs = new IntersectionObserver((entries) => {
      entries.forEach(e => {
        if (e.isIntersecting) {
          const img = e.target;
          img.src = img.dataset.src;
          img.removeAttribute('data-src');
          img.classList.add('loaded');
          obs.unobserve(img);
        }
      });
    }, { rootMargin: '200px' });
    imgs.forEach(img => obs.observe(img));
  }
  lazyLoadImages();

  // ===== Animate sections on scroll =====
  function animateSections() {
    const els = document.querySelectorAll('.animate-in:not(.animated)');
    if (!els.length) return;
    const obs = new IntersectionObserver((entries) => {
      entries.forEach(e => {
        if (e.isIntersecting) {
          e.target.classList.add('animated');
          obs.unobserve(e.target);
        }
      });
    }, { threshold: 0.1, rootMargin: '0px 0px -40px 0px' });
    els.forEach(el => obs.observe(el));
  }
  animateSections();

  // ===== Toast notifications =====
  window.showToast = function (msg, type = 'info') {
    let container = document.querySelector('.toast-container');
    if (!container) {
      container = document.createElement('div');
      container.className = 'toast-container';
      document.body.appendChild(container);
    }
    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.textContent = msg;
    container.appendChild(toast);
    setTimeout(() => {
      toast.style.opacity = '0';
      toast.style.transform = 'translateX(30px)';
      toast.style.transition = 'all 0.3s ease';
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  };

  // ===== Hero auto-rotate =====
  const heroItems = window.heroItems || [];
  if (heroItems.length > 1) {
    let heroIdx = 0;
    const heroBg = document.getElementById('heroBg');
    const heroTitle = document.getElementById('heroTitle');
    const heroDesc = document.getElementById('heroDesc');
    const heroPlayBtn = document.getElementById('heroPlayBtn');
    const heroInfoBtn = document.getElementById('heroInfoBtn');
    const heroBadge = document.getElementById('heroBadge');

    function setHero(idx) {
      const item = heroItems[idx];
      if (!item) return;
      if (heroBg) heroBg.style.backgroundImage = `url('${item.cover}')`;
      if (heroTitle) heroTitle.textContent = item.title;
      if (heroDesc) heroDesc.textContent = item.desc || '';
      if (heroPlayBtn) heroPlayBtn.href = `detail.php?id=${item.id}`;
      if (heroInfoBtn) heroInfoBtn.href = `detail.php?id=${item.id}`;
    }

    setInterval(() => {
      heroIdx = (heroIdx + 1) % heroItems.length;
      setHero(heroIdx);
    }, 7000);
  }

  // ===== TikTok-style Video Player =====
  window.BonifixPlayer = (function () {
    let episodes = [];
    let currentEpIndex = 0;
    let dramaTitle = '';
    let totalEps = 0;

    const player = document.getElementById('tiktokPlayer');
    const video = document.getElementById('tiktokVideo');
    const closeBtn = document.getElementById('tiktokClose');
    const epCurrent = document.getElementById('tiktokEpCurrent');
    const epTotalEl = document.getElementById('tiktokEpTotal');
    const titleEl = document.getElementById('tiktokTitle');
    const progressFill = document.getElementById('tiktokProgressFill');
    const progressBar = document.getElementById('tiktokProgressBar');
    const epListBtn = document.getElementById('tiktokEpListBtn');
    const nextOverlay = document.getElementById('nextEpOverlay');
    const nextCountdown = document.getElementById('nextCountdown');

    if (!player || !video) return {};

    function open(eps, idx, title) {
      episodes = eps;
      currentEpIndex = idx;
      dramaTitle = title;
      totalEps = eps.length;
      player.classList.add('active');
      document.body.style.overflow = 'hidden';
      loadEpisode(idx);
      if (epTotalEl) epTotalEl.textContent = totalEps;
      if (titleEl) titleEl.textContent = title;
    }

    function close() {
      player.classList.remove('active');
      document.body.style.overflow = '';
      if (video) { video.pause(); video.src = ''; }
      if (nextOverlay) nextOverlay.style.display = 'none';
    }

    function loadEpisode(idx) {
      if (idx < 0 || idx >= episodes.length) return;
      currentEpIndex = idx;
      const ep = episodes[idx];
      if (!ep) return;
      if (epCurrent) epCurrent.textContent = idx + 1;
      if (nextOverlay) nextOverlay.style.display = 'none';
      if (progressFill) progressFill.style.width = '0%';

      // Highlight active ep card in episode list
      document.querySelectorAll('.ep-card').forEach((c, i) => c.classList.toggle('active', i === idx));

      // Kumpulkan semua URL untuk dicoba (utama + alt fallback)
      const primarySrc = ep.mp4_hd || ep.mp4 || ep.mp4_fhd || ep.m3u8 || ep.url || '';
      const altSrcs    = ep.alt || [];
      const allSrcs    = primarySrc ? [primarySrc, ...altSrcs] : [...altSrcs];

      video.src = '';
      video.load();

      if (allSrcs.length) {
        // Coba satu per satu sampai ada yang berhasil (ada audio/video)
        if (typeof tryPlayUrls === 'function') {
          tryPlayUrls(video, allSrcs,
            function(goodUrl) {
              // Simpan URL valid untuk pemakaian berikutnya
              ep.mp4 = goodUrl; ep.mp4_hd = goodUrl; ep.mp4_fhd = goodUrl;
              if (titleEl) titleEl.textContent = dramaTitle;
            },
            function() {
              // Semua URL gagal — fetch dari server
              if (typeof fetchVideoUrl === 'function') {
                if (titleEl) titleEl.textContent = 'Memuat Episode ' + (idx + 1) + '...';
                fetchVideoUrl(idx, function(fetchedUrl) {
                  if (!fetchedUrl) {
                    if (titleEl) titleEl.textContent = dramaTitle;
                    showToast('Video episode ' + (idx + 1) + ' belum tersedia.', 'error');
                    return;
                  }
                  if (titleEl) titleEl.textContent = dramaTitle;
                  video.src = fetchedUrl;
                  video.play().catch(() => {});
                });
              } else {
                showToast('Video tidak tersedia untuk episode ini.', 'error');
              }
            }
          );
        } else {
          // Fallback jika tryPlayUrls belum tersedia
          video.src = allSrcs[0];
          video.play().catch(() => {});
        }
      } else {
        // Lazy-fetch URL video via fungsi yang didefinisikan di detail.php
        if (typeof fetchVideoUrl === 'function') {
          if (titleEl) titleEl.textContent = 'Memuat Episode ' + (idx + 1) + '...';
          fetchVideoUrl(idx, function(fetchedUrl) {
            if (!fetchedUrl) {
              if (titleEl) titleEl.textContent = dramaTitle;
              showToast('Video episode ' + (idx + 1) + ' belum tersedia.', 'error');
              return;
            }
            if (titleEl) titleEl.textContent = dramaTitle;
            video.src = fetchedUrl;
            video.play().catch(() => {});
          });
        } else {
          showToast('Video tidak tersedia untuk episode ini.', 'error');
        }
      }
    }

    function next() {
      if (currentEpIndex < episodes.length - 1) {
        loadEpisode(currentEpIndex + 1);
      } else {
        // Check if there are more episodes that weren't loaded due to API limits
        const realTotal = (typeof window.dramaChapterCount === 'number' && window.dramaChapterCount > 0)
          ? window.dramaChapterCount
          : episodes.length;
        if (realTotal > episodes.length) {
          showToast('Episode ' + (currentEpIndex + 2) + '-' + realTotal + ' belum dapat dimuat dari sumber. Coba lagi nanti.', 'error');
        } else {
          showToast('Ini adalah episode terakhir! 🎉', 'success');
        }
      }
    }

    function prev() {
      if (currentEpIndex > 0) loadEpisode(currentEpIndex - 1);
    }

    // Video events
    if (video) {
      video.addEventListener('timeupdate', () => {
        if (video.duration && progressFill) {
          progressFill.style.width = ((video.currentTime / video.duration) * 100) + '%';
        }
        // Show next ep countdown when 10 seconds left (only if there's a truly next ep)
        if (video.duration && (video.duration - video.currentTime) <= 10 && (video.duration - video.currentTime) > 0) {
          if (currentEpIndex < episodes.length - 1 && nextOverlay) {
            nextOverlay.style.display = 'flex';
            if (nextCountdown) nextCountdown.textContent = Math.ceil(video.duration - video.currentTime);
          }
        }
      });

      video.addEventListener('ended', () => {
        if (currentEpIndex < episodes.length - 1) {
          setTimeout(() => next(), 800);
        } else {
          // At end of loaded batch — check real total
          const realTotal = (typeof window.dramaChapterCount === 'number' && window.dramaChapterCount > 0)
            ? window.dramaChapterCount
            : episodes.length;
          if (realTotal > episodes.length) {
            showToast('Episode ' + (currentEpIndex + 2) + '-' + realTotal + ' belum dapat dimuat dari sumber. Coba lagi nanti.', 'error');
          } else {
            showToast('Ini adalah episode terakhir! 🎉', 'success');
          }
        }
      });

      // Tap to play/pause
      video.addEventListener('click', () => {
        if (video.paused) video.play();
        else video.pause();
      });
    }

    if (closeBtn) closeBtn.addEventListener('click', close);

    // Progress bar seek
    if (progressBar) {
      progressBar.addEventListener('click', (e) => {
        const rect = progressBar.getBoundingClientRect();
        const ratio = (e.clientX - rect.left) / rect.width;
        if (video.duration) video.currentTime = ratio * video.duration;
      });
    }

    // Swipe gesture (mobile)
    let touchStartY = 0;
    let touchStartX = 0;
    if (player) {
      player.addEventListener('touchstart', (e) => {
        touchStartY = e.touches[0].clientY;
        touchStartX = e.touches[0].clientX;
      }, { passive: true });

      player.addEventListener('touchend', (e) => {
        const dy = touchStartY - e.changedTouches[0].clientY;
        const dx = touchStartX - e.changedTouches[0].clientX;
        if (Math.abs(dy) > Math.abs(dx) && Math.abs(dy) > 50) {
          if (dy > 0) next();
          else prev();
        }
      }, { passive: true });
    }

    // Episode list button
    if (epListBtn) {
      epListBtn.addEventListener('click', () => {
        close();
        const epSection = document.getElementById('episodesSection');
        if (epSection) { epSection.scrollIntoView({ behavior: 'smooth' }); }
      });
    }

    // Next ep overlay click
    if (nextOverlay) {
      document.getElementById('nextEpBtn')?.addEventListener('click', () => next());
      document.getElementById('cancelNextBtn')?.addEventListener('click', () => {
        nextOverlay.style.display = 'none';
      });
    }

    // Keyboard controls
    document.addEventListener('keydown', (e) => {
      if (!player.classList.contains('active')) return;
      if (e.key === 'ArrowRight' || e.key === 'ArrowDown') next();
      if (e.key === 'ArrowLeft' || e.key === 'ArrowUp') prev();
      if (e.key === 'Escape') close();
      if (e.key === ' ') { e.preventDefault(); video.paused ? video.play() : video.pause(); }
    });

    return { open, close, next, prev, loadEpisode };
  })();

  // ===== Desktop Video Player (for detail page) =====
  window.desktopPlay = function (url, epIdx) {
    const vid = document.getElementById('desktopVideo');
    if (!vid || !url) return;
    const ep = (typeof episodesData !== 'undefined') ? episodesData[epIdx] : null;
    const altUrls = ep ? (ep.alt || []) : [];
    const allUrls = url ? [url, ...altUrls] : altUrls;
    document.querySelectorAll('.ep-card').forEach((c, i) => c.classList.toggle('active', i === epIdx));
    const label = document.getElementById('currentEpLabel');

    if (typeof tryPlayUrls === 'function' && allUrls.length > 1) {
      tryPlayUrls(vid, allUrls,
        function(goodUrl) {
          if (ep) { ep.mp4 = goodUrl; ep.mp4_hd = goodUrl; ep.mp4_fhd = goodUrl; }
          if (label) label.textContent = 'Episode ' + (epIdx + 1);
        },
        function() {
          if (label) label.textContent = 'Video tidak tersedia';
          if (typeof showToast === 'function') showToast('Video tidak tersedia untuk episode ini.', 'error');
        }
      );
    } else {
      vid.src = url;
      vid.load();
      vid.play().catch(() => {});
    }
    if (label) label.textContent = 'Episode ' + (epIdx + 1);
    const section = document.getElementById('desktopVideoSection');
    if (section) section.scrollIntoView({ behavior: 'smooth', block: 'start' });
  };

  // ===== Auto-play next ep on desktop video end =====
  const desktopVideo = document.getElementById('desktopVideo');
  if (desktopVideo) {
    desktopVideo.addEventListener('ended', () => {
      const nextBtn = document.getElementById('nextEpDesktop');
      if (nextBtn) nextBtn.click();
    });
    desktopVideo.addEventListener('timeupdate', () => {
      const bar = document.getElementById('desktopProgress');
      if (bar && desktopVideo.duration) {
        bar.style.width = ((desktopVideo.currentTime / desktopVideo.duration) * 100) + '%';
      }
    });
  }

  // ===== Search form mobile =====
  const mobileSearchForm = document.getElementById('mobileSearchForm');
  if (mobileSearchForm) {
    mobileSearchForm.addEventListener('submit', (e) => {
      e.preventDefault();
      const q = mobileSearchForm.querySelector('input')?.value?.trim();
      if (q) window.location.href = `search.php?q=${encodeURIComponent(q)}`;
    });
  }

  // ===== Search form desktop =====
  const desktopSearchForm = document.getElementById('desktopSearchForm');
  if (desktopSearchForm) {
    desktopSearchForm.addEventListener('submit', (e) => {
      e.preventDefault();
      const q = desktopSearchForm.querySelector('input')?.value?.trim();
      if (q) window.location.href = `search.php?q=${encodeURIComponent(q)}`;
    });
  }

  // ===== Back to top =====
  const backToTop = document.getElementById('backToTop');
  if (backToTop) {
    window.addEventListener('scroll', () => {
      backToTop.style.opacity = window.scrollY > 400 ? '1' : '0';
      backToTop.style.pointerEvents = window.scrollY > 400 ? 'auto' : 'none';
    });
    backToTop.addEventListener('click', () => window.scrollTo({ top: 0, behavior: 'smooth' }));
  }

  // ===== Ripple effect on buttons =====
  document.querySelectorAll('.btn-play, .btn-red, .btn-info').forEach(btn => {
    btn.addEventListener('click', function (e) {
      const ripple = document.createElement('span');
      ripple.style.cssText = `position:absolute;width:10px;height:10px;background:rgba(255,255,255,0.3);border-radius:50%;pointer-events:none;transform:scale(1);animation:ripple 0.6s linear;left:${e.offsetX - 5}px;top:${e.offsetY - 5}px;`;
      this.style.position = 'relative';
      this.style.overflow = 'hidden';
      this.appendChild(ripple);
      setTimeout(() => ripple.remove(), 600);
    });
  });

  // ===== Watch History (localStorage + Firebase Firestore for logged-in users) =====
  window.WatchHistory = (function () {
    const KEY     = 'boniflix_history';
    const MAX     = 10;

    function get() {
      try { return JSON.parse(localStorage.getItem(KEY) || '[]'); }
      catch { return []; }
    }

    function save(item) {
      // item = { id, title, cover, ep, lastEp }
      if (!item || !item.id) return;
      // Always save to localStorage
      let list = get().filter(x => x.id !== item.id);
      list.unshift({ ...item, ts: Date.now() });
      if (list.length > MAX) list = list.slice(0, MAX);
      localStorage.setItem(KEY, JSON.stringify(list));
      // Also sync to Firestore if logged in
      if (window.BonifixAuth && window.BonifixAuth.isLoggedIn()) {
        window.BonifixAuth.saveHistory(item);
      }
    }

    function clear() {
      localStorage.removeItem(KEY);
      // Also clear Firestore if logged in
      if (window.BonifixAuth && window.BonifixAuth.isLoggedIn()) {
        window.BonifixAuth.clearHistory();
      }
      const section = document.getElementById('history-section');
      if (section) section.style.display = 'none';
    }

    async function render() {
      const section = document.getElementById('history-section');
      if (!section) return;

      let list = get(); // Start with localStorage

      // If logged in, merge with Firestore data
      if (window.BonifixAuth && window.BonifixAuth.isLoggedIn()) {
        try {
          const cloudList = await window.BonifixAuth.getHistory();
          if (cloudList && cloudList.length) {
            // Merge: cloud items override local by id, then sort by ts
            const merged = {};
            list.forEach(item => { if (item.id) merged[item.id] = item; });
            cloudList.forEach(item => {
              if (item.id) {
                const existing = merged[item.id];
                if (!existing || (item.ts && item.ts > (existing.ts || 0))) {
                  merged[item.id] = item;
                }
              }
            });
            list = Object.values(merged).sort((a, b) => (b.ts || 0) - (a.ts || 0)).slice(0, 20);
          }
        } catch(e) {
          // Firestore unavailable — just use localStorage
        }
      }

      if (!list.length) { section.style.display = 'none'; return; }

      section.style.display = 'block';
      const row = section.querySelector('.drama-row');
      if (!row) return;
      row.innerHTML = '';

      const base = '/dramabox-main/boniflix';
      list.forEach(item => {
        const timeAgo = _timeAgo(item.ts);
        const a = document.createElement('a');
        a.href = `${base}/detail.php?id=${item.id}`;
        a.className = 'drama-card history-card';
        a.innerHTML = `
          <img class="drama-card-img" src="${item.cover}" alt="${item.title}"
               loading="lazy" onerror="this.src='https://via.placeholder.com/160x240/1a1a1a/555?text=?'">
          <span class="history-time-badge">${timeAgo}</span>
          ${item.ep ? `<span class="badge-ep">${item.ep} Ep</span>` : ''}
          ${item.lastEp ? `<span class="history-ep-badge">Ep ${item.lastEp}</span>` : ''}
          <div class="drama-card-overlay">
            <div class="drama-card-title">${item.title}</div>
            <div class="drama-card-actions"><span class="card-btn play-btn">▶ Lanjut</span></div>
          </div>`;
        row.appendChild(a);
      });
    }

    function _timeAgo(ts) {
      const sec = Math.floor((Date.now() - ts) / 1000);
      if (sec < 60)    return 'Baru saja';
      if (sec < 3600)  return Math.floor(sec / 60) + ' mnt lalu';
      if (sec < 86400) return Math.floor(sec / 3600) + ' jam lalu';
      return Math.floor(sec / 86400) + ' hari lalu';
    }

    // Auto-render on DOM ready
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', render);
    } else {
      render();
    }

    return { save, get, clear, render };
  })();

  // Track drama-card clicks to save history
  document.addEventListener('click', function (e) {
    const card = e.target.closest('a.drama-card');
    if (!card || card.closest('#history-section')) return; // don't re-log from history itself
    const href  = card.getAttribute('href') || '';
    const idMatch = href.match(/id=([0-9]+)/);
    if (!idMatch) return;
    const id    = idMatch[1];
    const img   = card.querySelector('img');
    const cover = img ? img.src : '';
    const titleEl = card.querySelector('.drama-card-title');
    const title = titleEl ? titleEl.textContent.trim() : '';
    const epEl  = card.querySelector('.badge-ep');
    const ep    = epEl ? epEl.textContent.replace(' Ep','').trim() : '';
    if (id) window.WatchHistory.save({ id, title, cover, ep });
  });

})();
